#pragma once

#include "PCANBasicCLR.h"
#include "PCAN-ISO-TP-CLR_2004.h"
#include "PCAN-ISO-TP-CLR_2016.h"

namespace PcanIsoTpExample {

	using namespace System;
	using namespace System::ComponentModel;
	using namespace System::Collections;
	using namespace System::Collections::Generic;
	using namespace System::Windows::Forms;
	using namespace System::Data;
	using namespace System::Drawing;
	using namespace System::Text;

	using namespace Peak::Can::IsoTp;

	/// <summary>
	/// Represents a PCAN device
	/// </summary>
	enum class TPCANDevice : Byte
	{
		/// <summary>
		/// Undefined, unknown or not selected PCAN device value
		/// </summary>
		PCAN_NONE = 0,
		/// <summary>
		/// PCAN Non-Plug and Play devices. NOT USED WITHIN PCAN-Basic API
		/// </summary>
		PCAN_PEAKCAN = 1,
		/// <summary>
		/// PCAN-ISA, PCAN-PC/104, and PCAN-PC/104-Plus
		/// </summary>
		PCAN_ISA = 2,
		/// <summary>
		/// PCAN-Dongle
		/// </summary>
		PCAN_DNG = 3,
		/// <summary>
		/// PCAN-PCI, PCAN-cPCI, PCAN-miniPCI, and PCAN-PCI Express
		/// </summary>
		PCAN_PCI = 4,
		/// <summary>
		/// PCAN-USB and PCAN-USB Pro
		/// </summary>
		PCAN_USB = 5,
		/// <summary>
		/// PCAN-PC Card
		/// </summary>
		PCAN_PCC = 6,
		/// <summary>
		/// PCAN Virtual hardware. NOT USED WITHIN PCAN-Basic API
		/// </summary>
		PCAN_VIRTUAL = 7,
		/// <summary>
		/// PCAN Gateway devices
		/// </summary>
		PCAN_LAN = 8
	};

	/// <summary>
	/// Represents an item of a ComboBox
	/// </summary>
	public ref class ComboBoxItem
	{
		/// <summary>
		/// Real data object to add to the comboBox
		/// </summary>
		public: property Object^ Data;
		/// <summary>
		/// Text to display in the comboBox
		/// </summary>
		property String^ Text;
		/// <summary>
		/// Default constructor
		/// </summary>
		/// <param name="text">Text to display.</param>
		/// <param name="data">Real data object.</param>
		ComboBoxItem(String^ text, Object^ data)
		{
			Text = text;
			Data = data;
		}

		String^ ToString() override
		{
			return Text;
		}
	};

	/// <summary>
	/// A static class providing various fonctions to format ISO-TP data.
	/// </summary>
	public ref class CanTpUtils
	{
		/// <summary>
		/// Maximum data length of a physical ISO-TP message.
		/// </summary>
		public: static const int ISOTP_MSG_PHYS_MAX_LENGTH = 4095;
		/// <summary>
		/// Maximum data length of a functional ISO-TP message.
		/// </summary>
		static const int ISOTP_MSG_FUNC_MAX_LENGTH = 7;
		/// <summary>
		/// Gets the formated text from a PCAN-ISO-TP Btr0/Btr1 code.
		/// </summary>
		/// <param name="handle">PCAN-ISO-TP Baudrate to format</param>
		/// <returns>The formatted text for a baudrate</returns>
		static String^ GetBitrate(TPCANTPBaudrate value)
		{
			switch (value)
			{
			case TPCANTPBaudrate::PCANTP_BAUD_1M:
				return "1 MBit/s";
			case TPCANTPBaudrate::PCANTP_BAUD_800K:
				return "800 kBit/s";
			case TPCANTPBaudrate::PCANTP_BAUD_500K:
				return "500 kBit/s";
			case TPCANTPBaudrate::PCANTP_BAUD_250K:
				return "250 kBit/s";
			case TPCANTPBaudrate::PCANTP_BAUD_125K:
				return "125 kBit/s";
			case TPCANTPBaudrate::PCANTP_BAUD_100K:
				return "100 kBit/s";
			case TPCANTPBaudrate::PCANTP_BAUD_95K:
				return "95,238 kBit/s";
			case TPCANTPBaudrate::PCANTP_BAUD_83K:
				return "83,333 kBit/s";
			case TPCANTPBaudrate::PCANTP_BAUD_50K:
				return "50 kBit/s";
			case TPCANTPBaudrate::PCANTP_BAUD_47K:
				return "47,619 kBit/s";
			case TPCANTPBaudrate::PCANTP_BAUD_33K:
				return "33,333 kBit/s";
			case TPCANTPBaudrate::PCANTP_BAUD_20K:
				return "20 kBit/s";
			case TPCANTPBaudrate::PCANTP_BAUD_10K:
				return "10 kBit/s";
			case TPCANTPBaudrate::PCANTP_BAUD_5K:
				return "5 kBit/s";
			}
			return Convert::ToString(value);
		}
		/// <summary>
		/// Gets the formated text for a PCAN-ISO-TP channel handle.
		/// </summary>
		/// <param name="handle">PCAN-ISO-TP Handle to format</param>
		/// <returns>The formatted text for a channel</returns>
		static String^ GetChannelName(TPCANTPHandle ^handle)
		{
			TPCANDevice devDevice;
			Byte byChannel;

			// Gets the owner device and channel for a 
			// PCAN-Basic handle
			if ((UInt16)handle < 0x100)
			{
				devDevice = (TPCANDevice)((UInt16)handle >> 4);
				byChannel = (Byte)((UInt16)handle & 0xF);
			}
			else
			{
				devDevice = (TPCANDevice)((UInt16)handle >> 8);
				byChannel = (Byte)((UInt16)handle & 0xFF);
			}
			// Constructs the PCAN-Basic Channel name and return it
			return GetPcanDevice(devDevice) + " " + String::Format("{0} ({1:X1}h)", byChannel, handle);
		}
		/// <summary>
		/// Returns string corresponding to the given TPCANDevice
		/// </summary> 
		static String^ GetPcanDevice(TPCANDevice dev) {
			switch (dev) {
			case TPCANDevice::PCAN_NONE:
				return "PCAN_NONE";
			case TPCANDevice::PCAN_PEAKCAN:
				return "PCAN_PEAKCAN";
			case TPCANDevice::PCAN_ISA:
				return "PCAN_ISA";
			case TPCANDevice::PCAN_DNG:
				return "PCAN_DNG";
			case TPCANDevice::PCAN_PCI:
				return "PCAN_PCI";
			case TPCANDevice::PCAN_USB:
				return "PCAN_USB";
			case TPCANDevice::PCAN_PCC:
				return "PCAN_PCC";
			case TPCANDevice::PCAN_VIRTUAL:
				return "PCAN_VIRTUAL";
			case TPCANDevice::PCAN_LAN:
				return "PCAN_LAN";
			}
			return "UNKNOWN";
		}
		/// <summary>
		/// Gets the formated text from a TPCANTPParameter value.
		/// </summary>
		/// <param name="handle">Parameter to format.</param>
		/// <returns>The parameter as a text.</returns>
		static String^ GetParameter(TPCANTPParameter param)
		{
			switch (param)
			{
			case TPCANTPParameter::PCANTP_PARAM_API_VERSION:
				return "API version";
			case TPCANTPParameter::PCANTP_PARAM_BLOCK_SIZE:
				return "ISO-TP Block Size (BS)";
			case TPCANTPParameter::PCANTP_PARAM_CAN_DATA_PADDING:
				return "CAN Data Padding";
			case TPCANTPParameter::PCANTP_PARAM_CHANNEL_CONDITION:
				return "Channel condition";
			case TPCANTPParameter::PCANTP_PARAM_DEBUG:
				return "Debug mode";
			case TPCANTPParameter::PCANTP_PARAM_MSG_PENDING:
				return "Message pending notification";
			case TPCANTPParameter::PCANTP_PARAM_PADDING_VALUE:
				return "CAN Data Padding value";
			case TPCANTPParameter::PCANTP_PARAM_RECEIVE_EVENT:
				return "Receive event";
			case TPCANTPParameter::PCANTP_PARAM_SEPARATION_TIME:
				return "ISO-TP Separation time (STMin)";
			case TPCANTPParameter::PCANTP_PARAM_WFT_MAX:
				return "ISO-TP FC::Wait frame max:: (N_WFTmax)";
			case TPCANTPParameter::PCANTP_PARAM_J1939_PRIORITY:
				return "ISO-TP J1939 priority";
			case TPCANTPParameter::PCANTP_PARAM_CAN_TX_DL:
				return "ISO-TP FD Data Length Code (DLC)";
			}
			return "Unknown parameter";
		}
		/// <summary>
		/// Gets the formatted text of a TPCANTPStatus error.
		/// </summary>
		/// <param name="error">Error code to be translated</param>
		/// <returns>A text with the translated error</returns>
		static String^ GetError(TPCANTPStatus error)
		{
			StringBuilder ^strTemp;

			// Creates a buffer big enough for a error-text
			strTemp = gcnew StringBuilder(256);
			// Gets the text using the GetErrorText API function
			// If the function is successful, the translated error is returned. 
			// If it fails, a text describing the current error is returned.
			if (CanTpApi::GetErrorText(error, 0, strTemp) != TPCANTPStatus::PCANTP_ERROR_OK)
				return String::Format("An error occurred. Error-code's text ({0:X}) couldn't be retrieved", (unsigned int)error);
			else
				return strTemp->ToString();
		}
		/// <summary>
		/// Gets the formated text of a TPCANTPFormatType value.
		/// </summary>
		/// <param name="format">value to format</param>
		/// <returns>The parameter formatted as a human-readable string.</returns>
		static String^ GetFormatType(TPCANTPFormatType format)
		{
			switch (format)
			{
			case TPCANTPFormatType::PCANTP_FORMAT_ENHANCED:
				return "Enhanced";
			case TPCANTPFormatType::PCANTP_FORMAT_EXTENDED:
				return "Extended";
			case TPCANTPFormatType::PCANTP_FORMAT_FIXED_NORMAL:
				return "Fixed Normal";
			case TPCANTPFormatType::PCANTP_FORMAT_MIXED:
				return "Mixed";
			case TPCANTPFormatType::PCANTP_FORMAT_NORMAL:
				return "Normal";
			case TPCANTPFormatType::PCANTP_FORMAT_UNKNOWN:
			default:
				return "Unknown";
			}
		}
		/// <summary>
		/// Gets the formated text of a timestamp.
		/// </summary>
		/// <param name="ts">Timestamp to format</param>
		/// <param name="showPeriod">States if the output is the period between 2 timestamps.</param>
		/// <param name="tsOld">If showPeriod is true, the period is based on that previous timestamp</param>
		/// <returns>The parameter formatted as a human-readable string.</returns>
		static String^ GetTimeString(TPCANTPTimestamp ts, bool showPeriod, TPCANTPTimestamp tsOld)
		{
			double fTime;

			fTime = ts.millis + (ts.micros / 1000.0);
			if (showPeriod)
				fTime -= (tsOld.millis + (tsOld.micros / 1000.0));
			return fTime.ToString("F1");
		}
		/// <summary>
		/// Gets the data of an ISO-TP message as a formatted string.
		/// </summary>
		/// <param name="msg">ISO-TP message holding the data to format.</param>
		/// <returns>The parameter formatted as a human-readable string.</returns>
		static String^ GetDataString(TPCANTPMsg msg)
		{
			String ^strTemp;

			strTemp = "";
			for (int i = 0; i < msg.LEN; i++)
				strTemp += String::Format("{0:X2} ", msg.DATA[i]);
			return strTemp;
		}
		/// <summary>
		/// Gets a Unique Identifier based of an ISO-TP message as a formatted string.
		/// </summary>
		/// <param name="msg">The ISO-TP message.</param>
		/// <returns>The parameter formatted as a human-readable string.</returns>
		static String^ GetUniqueId(TPCANTPMsg msg)
		{
			// We format the ID of the message and show it
			return String::Format("{0:X2}h_{1:X2}h_{2}_{3}{4}_{5}_{6:X2}h",
				msg.SA, msg.TA,
				GetCanIdType(msg.IDTYPE),
				GetTargetType(msg.TA_TYPE),
				GetMsgType(msg.MSGTYPE), GetFormatType(msg.FORMAT), msg.RA);
		}
		/// <summary>
		/// Gets the CAN ID type as a formatted string.
		/// </summary>
		/// <param name="type">The can id type to format.</param>
		/// <returns>The parameter formatted as a human-readable string.</returns>
		static String^ GetCanIdType(TPCANTPIdType type)
		{
			String^ strIdTypeReturn = "";
			if ((type & TPCANTPIdType::PCANTP_ID_CAN_11BIT) == TPCANTPIdType::PCANTP_ID_CAN_11BIT)
				strIdTypeReturn = "11bits ";
			if ((type & TPCANTPIdType::PCANTP_ID_CAN_29BIT) == TPCANTPIdType::PCANTP_ID_CAN_29BIT)
				strIdTypeReturn += "29bits ";
			if ((type & TPCANTPIdType::PCANTP_ID_CAN_FD) == TPCANTPIdType::PCANTP_ID_CAN_FD)
				strIdTypeReturn += "FD ";
			if ((type & TPCANTPIdType::PCANTP_ID_CAN_BRS) == TPCANTPIdType::PCANTP_ID_CAN_BRS)
				strIdTypeReturn += "BRS ";
			if (strIdTypeReturn == "")
				strIdTypeReturn = "Unknown";
			return strIdTypeReturn;
		}
		/// <summary>
		/// Gets the ISO-TP Addressing type as a formatted string.
		/// </summary>
		/// <param name="type">The addressing type to format.</param>
		/// <returns>The parameter formatted as a human-readable string.</returns>
		static String^ GetTargetType(TPCANTPAddressingType type)
		{
			switch (type)
			{
			case TPCANTPAddressingType::PCANTP_ADDRESSING_FUNCTIONAL:
				return "Functional";
			case TPCANTPAddressingType::PCANTP_ADDRESSING_PHYSICAL:
				return "Physical";
			case TPCANTPAddressingType::PCANTP_ADDRESSING_UNKNOWN:
			default:
				return "Unknown";
			}
		}
		/// <summary>
		/// Gets the ISO-TP message type as a formatted string.
		/// </summary>
		/// <param name="type">The message type to format.</param>
		/// <returns>The parameter formatted as a human-readable string.</returns>
		static String^ GetMsgType(TPCANTPMessageType type)
		{
			switch (type)
			{
			case TPCANTPMessageType::PCANTP_MESSAGE_DIAGNOSTIC:
				return "Diagnostic";
			case TPCANTPMessageType::PCANTP_MESSAGE_INDICATION:
				return "Indication";
			case TPCANTPMessageType::PCANTP_MESSAGE_REMOTE_DIAGNOSTIC:
				return "Remote Diagnostic";
			case TPCANTPMessageType::PCANTP_MESSAGE_REQUEST_CONFIRMATION:
				return "Confirmation";
			case TPCANTPMessageType::PCANTP_MESSAGE_INDICATION_TX:
				return "Indication TX";
			case TPCANTPMessageType::PCANTP_MESSAGE_UNKNOWN:
			default:
				return "Unknown";
			}
		}
		/// <summary>
		/// Gets an ISO-TP address type as a formatted string.
		/// </summary>
		/// <param name="type">The address to format.</param>
		/// <returns>The parameter formatted as a human-readable string.</returns>
		static String^ GetAddress(Byte address)
		{
			return String::Format("{0:X2}h", address);
		}
		/// <summary>
		/// Gets an CAN ID as a formatted string.
		/// </summary>
		/// <param name="canId">The CAN ID to format.</param>
		/// <param name="isExtended">True if the CAN ID is 29bits.</param>
		/// <returns>The parameter formatted as a human-readable string.</returns>
		static String^ GetCanId(UInt32 canId, bool isExtended)
		{
			if (!isExtended)
				return String::Format("{0:X3}h", canId);
			else
				return String::Format("{0:X8}h", canId);
		}
		/// <summary>
		/// Gets an ISO-TP Network Result as a formatted string.
		/// </summary>
		/// <param name="result">The result to format.</param>
		/// <returns>The parameter formatted as a human-readable string.</returns>
		static String^ GetResult(TPCANTPConfirmation result)
		{
			String ^strTmp;

			switch (result)
			{
			case TPCANTPConfirmation::PCANTP_N_BUFFER_OVFLW:
				strTmp = "Buffer overflow";
				break;
			case TPCANTPConfirmation::PCANTP_N_ERROR:
				strTmp = "Error";
				break;
			case TPCANTPConfirmation::PCANTP_N_INVALID_FS:
				strTmp = "Invalid First frame";
				break;
			case TPCANTPConfirmation::PCANTP_N_OK:
				strTmp = "Ok";
				break;
			case TPCANTPConfirmation::PCANTP_N_TIMEOUT_A:
				strTmp = "Timeout A";
				break;
			case TPCANTPConfirmation::PCANTP_N_TIMEOUT_BS:
				strTmp = "Timeout Bs";
				break;
			case TPCANTPConfirmation::PCANTP_N_TIMEOUT_CR:
				strTmp = "Timeout Cr";
				break;
			case TPCANTPConfirmation::PCANTP_N_UNEXP_PDU:
				strTmp = "Unexpected Protocol Data Unit";
				break;
			case TPCANTPConfirmation::PCANTP_N_WFT_OVRN:
				strTmp = "Wait For Transmit overrun";
				break;
			case TPCANTPConfirmation::PCANTP_N_WRONG_SN:
				strTmp = "Wrong Sequence Number";
				break;
			default:
				strTmp = "Unknown";
				break;
			}
			return String::Format("{0} ({1})", (Byte)result, strTmp);
		}
	};

	/// <summary>
	/// Class that stores information of a single ISO-TP mapping.
	/// </summary>
	public ref class MappingStatus
	{
		/// <summary>
		/// CAN ID to map to CAN-ISO-TP network addressing information
		/// </summary>
		public: property UInt32 m_canId;
		/// <summary>
		/// CAN ID used by the other side to internally respond to the CAN-ISO-TP segmented frames
		/// (i.e. the Flow Control frames will use this ID)
		/// </summary>
		property UInt32 m_canIdResponse;
		/// <summary>
		/// The CAN ID type used by the mapping (11 bits or 29 bits CAN ID)
		/// </summary>
		property TPCANTPIdType m_canIdType;
		/// <summary>
		/// The ISO-TP network addressing format.
		/// </summary>
		property TPCANTPFormatType m_formatType;
		/// <summary>
		/// Type of CAN-ISO-TP message (diagnostic or remote disgnostic message)
		/// </summary>
		property TPCANTPMessageType m_msgType;
		/// <summary>
		/// Source address
		/// </summary>
		property Byte m_sourceAddr;
		/// <summary>
		/// Target address
		/// </summary>
		property Byte m_targetAddr;
		/// <summary>
		/// Type of addressing (physical: -node to node-, or functional: -node to all-)
		/// </summary>
		property TPCANTPAddressingType m_targetType;
		/// <summary>
		/// Remote address (used only with remote disgnostic message)
		/// </summary>
		property Byte m_remoteAddr;
		/// <summary>
		/// Default constructor
		/// </summary>
		/// <param name="canId">CAN ID to map to CAN-ISO-TP network addressing information</param>
		/// <param name="canIdResponse">CAN ID used by the other side to internally respond to the CAN-ISO-TP segmented frames
		/// (i.e. the Flow Control frames will use this ID)</param>
		/// <param name="canIdType">The CAN ID type used by the mapping (11 bits or 29 bits CAN ID)</param>
		/// <param name="formatType">The ISO-TP network addressing format.</param>
		/// <param name="msgType">Type of CAN-ISO-TP message (diagnostic or remote disgnostic message)</param>
		/// <param name="sourceAddr">Source address</param>
		/// <param name="targetAddr">Target address</param>
		/// <param name="targetType">Type of addressing (physical: -node to node-, or functional: -node to all-)</param>
		/// <param name="remoteAddr">Remote address (used only with remote disgnostic message)</param>
		MappingStatus(
			UInt32 canId,
			UInt32 canIdResponse,
			TPCANTPIdType canIdType,
			TPCANTPFormatType formatType,
			TPCANTPMessageType msgType,
			TPCANTPAddressingType targetType,
			Byte sourceAddr,
			Byte targetAddr,
			Byte remoteAddr)
		{
			m_canId = canId;
			m_canIdResponse = canIdResponse;
			m_canIdType = canIdType;
			m_formatType = formatType;
			m_msgType = msgType;
			m_sourceAddr = sourceAddr;
			m_targetAddr = targetAddr;
			m_targetType = targetType;
			m_remoteAddr = remoteAddr;
		}
		/// <summary>
		/// Overload constructor
		/// </summary>
		MappingStatus() {
			m_canId = 0;
			m_canIdResponse = 0;
			m_canIdType = TPCANTPIdType::PCANTP_ID_CAN_11BIT;
			m_formatType = TPCANTPFormatType::PCANTP_FORMAT_UNKNOWN;
			m_msgType = TPCANTPMessageType::PCANTP_MESSAGE_UNKNOWN;
			m_sourceAddr = 0x00;
			m_targetAddr = 0x00;
			m_targetType = TPCANTPAddressingType::PCANTP_ADDRESSING_UNKNOWN;
			m_remoteAddr = 0x00;
		}
		/// <summary>
		/// Gets a new instance of the object.
		/// </summary>
		/// <returns>The cloned instance of the object.</returns>
		MappingStatus^ Clone()
		{
			return gcnew MappingStatus(m_canId, m_canIdResponse, m_canIdType,
				m_formatType, m_msgType, m_targetType, m_sourceAddr, m_targetAddr, m_remoteAddr);
		}
		/// <summary>
		/// States if an ISO-TP mapping is NOT required for this configuration.
		///  Some 29 bits CAN ID ISO-TP message with specific format addressing 
		///  do NOT require ISO-TP mappings.
		/// </summary>
		property bool IsAutomatic
		{
			bool get()
			{
				if (m_canIdType == TPCANTPIdType::PCANTP_ID_CAN_29BIT && (
					m_formatType == TPCANTPFormatType::PCANTP_FORMAT_FIXED_NORMAL ||
					m_formatType == TPCANTPFormatType::PCANTP_FORMAT_MIXED ||
					m_formatType == TPCANTPFormatType::PCANTP_FORMAT_ENHANCED)
					)
					return true;
				return false;
			}
		}
		/// <summary>
		/// The formatted name of this mapping configuration.
		/// </summary>
		property String^ Name
		{
			String^ get()
			{
				String^ name;

				name = CanIdType + ", " + FormatType + ", " + TargetType;
				if (!IsAutomatic)
					name += ": " + SourceAddress + " -> " + TargetAddress;
				return name;
			}
		}
		/// <summary>
		/// The CAN ID of the configured mapping as a string.
		/// Note: automatic mapping has no CAN ID, since it is 
		/// handled on the fly by the ISO-TP API.
		/// </summary>
		property String^ CanId
		{
			String^ get()
			{
				if (IsAutomatic)
					return "-";
				return CanTpUtils::GetCanId(m_canId, m_canIdType == TPCANTPIdType::PCANTP_ID_CAN_29BIT);
			}
		}
		/// <summary>
		/// The CAN ID response of the configured mapping as a string.
		/// Note: automatic mapping has no CAN ID response, since it is 
		/// handled on the fly by the ISO-TP API.
		/// </summary>
		property String^ CanIdResponse
		{
			String^ get()
			{
				if (IsAutomatic)
					return "-";
				return CanTpUtils::GetCanId(m_canIdResponse, m_canIdType == TPCANTPIdType::PCANTP_ID_CAN_29BIT);
			}
		}
		/// <summary>
		/// The CAN ID type of the configured mapping as a string.
		/// </summary>
		property String^ CanIdType
		{
			String^ get() { return CanTpUtils::GetCanIdType(m_canIdType); }
		}
		/// <summary>
		/// The ISO-TP message type of the configured mapping as a string.
		/// </summary>
		property String^ MsgType
		{
			String^ get() { return CanTpUtils::GetMsgType(m_msgType); }
		}
		/// <summary>
		/// The ISO-TP addressing format type of the configured mapping as a string.
		/// </summary>
		property String^ FormatType
		{
			String^ get() { return CanTpUtils::GetFormatType(m_formatType); }
		}
		/// <summary>
		/// The ISO-TP target address type ID type of the configured mapping as a string.
		/// </summary>
		property String^ TargetType
		{
			String^ get() { return CanTpUtils::GetTargetType(m_targetType); }
		}
		/// <summary>
		/// The source address of the configured mapping as a string.
		/// </summary>
		property String^ SourceAddress
		{
			String^ get()
			{
				if (IsAutomatic)
					return "-";
				return CanTpUtils::GetAddress(m_sourceAddr);
			}
		}
		/// <summary>
		/// The target address of the configured mapping as a string.
		/// </summary>
		property String^ TargetAddress
		{
			String^ get()
			{
				if (IsAutomatic)
					return "-";
				return CanTpUtils::GetAddress(m_targetAddr);
			}
		}
		/// <summary>
		/// The remote address of the configured mapping as a string.
		/// </summary>
		property String^ RemoteAddress
		{
			String^ get()
			{
				if (IsAutomatic)
					return "-";
				return CanTpUtils::GetAddress(m_remoteAddr);
			}
		}
	};

	/// <summary>
	/// Class that stores information on a received ISO-TP message.
	/// </summary>
	public ref class MessageStatus
	{
		/// <summary>
		/// The last ISO-TP message received.
		/// </summary>
		private: TPCANTPMsg m_msg;
		/// <summary>
		/// The timestamp of the last ISO-TP message received.
		/// </summary>
		TPCANTPTimestamp m_timeStamp;
		/// <summary>
		/// The timestamp of the before last ISO-TP message received.
		/// </summary>
		TPCANTPTimestamp m_timeStampOld;
		/// <summary>
		/// Index of the object in the message listView.
		/// </summary>
		int m_index;
		/// <summary>
		/// Number of similar ISO-TP message received 
		/// (i.e. messages having the same Network Address Information).
		/// </summary>
		int m_count;
		/// <summary>
		/// States if the timestamp should be displayed as a period.
		/// </summary>
		bool m_showPeriod;
		/// <summary>
		/// States if the object has been modified but not updated/displayed on the UI.
		/// </summary>
		bool m_wasChanged;
		/// <summary>
		/// Default constructor.
		/// </summary>
		/// <param name="canTpMsg">The received ISO-TP message.</param>
		/// <param name="canTpTimestamp">The timestamp when the message was received.</param>
		/// <param name="listIndex">Position of the messageStatus in the message listView.</param>
		public: MessageStatus(TPCANTPMsg canTpMsg, TPCANTPTimestamp canTpTimestamp, int listIndex)
		{
			m_msg = canTpMsg;
			m_timeStamp = canTpTimestamp;
			m_timeStampOld = canTpTimestamp;
			m_index = listIndex;
			m_count = 1;
			m_showPeriod = true;
			m_wasChanged = false;
		}
		/// <summary>
		/// Updates the messageStatus' information with a newly received ISO-TP message.
		/// </summary>
		/// <param name="canTpMsg">The received ISO-TP message.</param>
		/// <param name="canTpTimestamp">The timestamp when the message was received.</param>
		void update(TPCANTPMsg canTpMsg, TPCANTPTimestamp canTpTimestamp)
		{
			m_msg = canTpMsg;
			m_timeStampOld = m_timeStamp;
			m_timeStamp = canTpTimestamp;
			m_wasChanged = true;
			m_count += 1;
		}
		/// <summary>
		/// States if a message has the same network address information as 
		/// the last ISO-TP message received.
		/// </summary>
		/// <param name="canTpMsg">The ISO-TP message to compare to.</param>
		/// <returns></returns>
		bool isSameNetAddrInfo(TPCANTPMsg canTpMsg)
		{
			return (m_msg.SA == canTpMsg.SA &&
				m_msg.TA == canTpMsg.TA &&
				m_msg.RA == canTpMsg.RA &&
				m_msg.IDTYPE == canTpMsg.IDTYPE &&
				m_msg.FORMAT == canTpMsg.FORMAT &&
				m_msg.TA_TYPE == canTpMsg.TA_TYPE &&
				m_msg.MSGTYPE == canTpMsg.MSGTYPE);
		}

#pragma region Getters / Setters
		/// <summary>
		/// The last ISO-TP message received.
		/// </summary>
		property TPCANTPMsg CanTpMsg
		{
			TPCANTPMsg get() { return m_msg; }
		}
		/// <summary>
		/// The timestamp of the last ISO-TP message received.
		/// </summary>
		property TPCANTPTimestamp Timestamp
		{
			TPCANTPTimestamp get() { return m_timeStamp; }
		}
		/// <summary>
		/// The index of the object in the "received message" listView.
		/// </summary>
		property int Position
		{
			int get() { return m_index; }
		}
		/// <summary>
		/// Number of time a similar message was received (same network address information).
		/// </summary>
		property int Count
		{
			int get() { return m_count; }
		}
		/// <summary>
		/// States if the timestamp should be displayed as a period.
		/// </summary>
		property bool ShowingPeriod
		{
			bool get() { return m_showPeriod; }
			void set(bool value)
			{
				if (m_showPeriod ^ value)
				{
					m_showPeriod = value;
					m_wasChanged = true;
				}
			}
		}
		/// <summary>
		/// States if the object was modified since last display.
		/// </summary>
		property bool MarkedAsUpdated
		{
			bool get() { return m_wasChanged; }
			void set(bool value) { m_wasChanged = value; }
		}
		/// <summary>
		/// Unique ID as a string of the last message received.
		/// </summary>
		property String^ UID
		{
			String^ get() { return CanTpUtils::GetUniqueId(m_msg); }
		}
		/// <summary>
		/// CAN ID type as a string of the last message received.
		/// </summary>
		property String^ CanIdType
		{
			String^ get() { return CanTpUtils::GetCanIdType(m_msg.IDTYPE); }
		}
		/// <summary>
		/// ISO-TP Message type as a string of the last message received.
		/// </summary>
		property String^ MsgType
		{
			String^ get() { return CanTpUtils::GetMsgType(m_msg.MSGTYPE); }
		}
		/// <summary>
		/// ISO-TP addressing format type as a string of the last message received.
		/// </summary>
		property String^ FormatType
		{
			String^ get() { return CanTpUtils::GetFormatType(m_msg.FORMAT); }
		}
		/// <summary>
		/// ISO-TP target addressing type as a string of the last message received.
		/// </summary>
		property String^ TargetType
		{
			String^ get() { return CanTpUtils::GetTargetType(m_msg.TA_TYPE); }
		}
		/// <summary>
		/// ISO-TP source address as a string of the last message received.
		/// </summary>
		property String^ SourceAddress
		{
			String^ get()
			{
				if (m_msg.FORMAT == TPCANTPFormatType::PCANTP_FORMAT_ENHANCED)
					// Unique SA is 11bits: store bits [7..0] in SA
					// and the rest in [3..0] of RA
					return CanTpUtils::GetCanId(((UInt32)((m_msg.RA >> 4) & 0x7)) << 8 | m_msg.SA, false);
				return CanTpUtils::GetAddress(m_msg.SA);
			}
		}
		/// <summary>
		/// ISO-TP target address as a string of the last message received.
		/// </summary>
		property String^ TargetAddress
		{
			String^ get()
			{
				if (m_msg.FORMAT == TPCANTPFormatType::PCANTP_FORMAT_ENHANCED)
					// Unique TA is 11bits: store bits [7..0] in TA
					// and the rest in [7..4] of RA
					return CanTpUtils::GetCanId(((UInt32)(m_msg.RA & 0x7)) << 8 | m_msg.TA, false);
				return CanTpUtils::GetAddress(m_msg.TA);
			}
		}
		/// <summary>
		/// ISO-TP remote address as a string of the last message received.
		/// </summary>
		property String^ RemoteAddress
		{
			String^ get() { return CanTpUtils::GetAddress(m_msg.RA); }
		}
		/// <summary>
		/// Lenght of the data as a string of the last message received.
		/// </summary>
		property String^ LengthString
		{
			String^ get() { return m_msg.LEN.ToString(); }
		}
		/// <summary>
		/// Data as a string of the last message received.
		/// </summary>
		property String^ DataString
		{
			String^ get() { return CanTpUtils::GetDataString(m_msg); }
		}
		/// <summary>
		/// ISO-TP Network result as a string of the last message received.
		/// </summary>
		property String^ ResultString
		{
			String^ get() { return CanTpUtils::GetResult(m_msg.RESULT); }
		}
		/// <summary>
		/// Timestamp (or period) as a string of the last message received.
		/// </summary>
		property String^ TimeString
		{
			String^ get() { return CanTpUtils::GetTimeString(m_timeStamp, m_showPeriod, m_timeStampOld); }
		}
#pragma endregion

	};
}